<?php

namespace Adfox\Blog\Livewire;

use Livewire\Component;

use App\Models\BlogComment;
use App\Models\BlogPost;
use App\Settings\BlogSettings;
use App\Settings\GeneralSettings;
use App\Settings\SEOSettings;
use Livewire\WithPagination;
use Artesaos\SEOTools\Traits\SEOTools as SEOToolsTrait;
use Livewire\Attributes\Url;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\MarkdownEditor;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Illuminate\Support\Str;
use Parsedown;

class PostComponent extends Component  implements HasForms
{
    use WithPagination, SEOToolsTrait;
    use InteractsWithForms;

    #[Url(as: 'ref',  except: '/')]
    public $referrer = '/blog';
    public ?array $data = [];
    public $post;
    public $descriptionHtml;

    public $name;
    public $email;
    public $comment;

    /**
     * Init component
     *
     * @return void
     */
    public function mount($slug)
    {
        // Check if blog system enabled
        if (!app(BlogSettings::class)->enable_blog) {
            return redirect('/');
        }

        // Check if user connected
        if (auth()->check()) {
            $this->form->fill([
                'name' => auth()->user()->name,
                'email' => auth()->user()->email
            ]);
        }

        // Get post
        $post = BlogPost::where('slug', $slug)->firstOrFail();

        // Set post
        $this->post = $post;

        $this->setDescriptionHtml($post->content);

        $this->setSEOData();
    }


    /**
     * Convert the ad description to HTML.
     */
    protected function setDescriptionHtml($content)
    {
        $parsedown = new Parsedown();
        $this->descriptionHtml = $parsedown->text($content);
    }


    public function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('name')
                    ->required(),
                TextInput::make('email')
                    ->required(),
                Textarea::make('comment')
                    ->columnSpanFull()
                    ->required(),
            ])
            ->columns(2)
            ->statePath('data');
    }

    public function addComment()
    {
        $formData = $this->form->getState();

        $formData['post_id'] = $this->post->id;

        BlogComment::create($formData);

        Notification::make()
            ->title(__('messages.t_command_has_been_posted'))
            ->success()
            ->send();

        $this->form->fill([
                'name' => auth()->user()->name,
                'email' => auth()->user()->email
            ]);
    }

    /**
     * Set SEO data
     */
    protected function setSeoData()
    {
        $generalSettings = app(GeneralSettings::class);
        $seoSettings = app(SEOSettings::class);

        $siteName = $generalSettings->site_name ?? app_name();
        $separator = $generalSettings->separator ?? '-';

        $title =  $this->post->title . " $separator " . $siteName;
        $description = Str::limit($this->post->content ?? app_name(), 160);
        $ogImage = getSettingMediaUrl('seo.ogimage', 'seo', asset('images/ogimage.jpg'));
        $this->seo()->setTitle($title);
        $this->seo()->setDescription($description);
        $this->seo()->setCanonical(url()->current());
        $this->seo()->opengraph()->setTitle($title);
        $this->seo()->opengraph()->setDescription($description);
        $this->seo()->opengraph()->setUrl(url()->current());
        $this->seo()->opengraph()->setType('website');
        $this->seo()->opengraph()->addImage($ogImage);
        $this->seo()->twitter()->setImage($ogImage);
        $this->seo()->twitter()->setUrl(url()->current());
        $this->seo()->twitter()->setSite("@" . $seoSettings->twitter_username);
        $this->seo()->twitter()->addValue('card', 'summary_large_image');
        $this->seo()->metatags()->addMeta('fb:page_id', $seoSettings->facebook_page_id, 'property');
        $this->seo()->metatags()->addMeta('fb:app_id', $seoSettings->facebook_app_id, 'property');
        $this->seo()->metatags()->addMeta('robots', 'index, follow, max-image-preview:large, max-snippet:-1, max-video-preview:-1', 'name');
        $this->seo()->jsonLd()->setTitle($title);
        $this->seo()->jsonLd()->setDescription($description);
        $this->seo()->jsonLd()->setUrl(url()->current());
        $this->seo()->jsonLd()->setType('WebSite');
    }


    /**
     * Get comments
     *
     * @return object
     */
    public function getCommentsProperty()
    {
        return BlogComment::where('post_id', $this->post->id)->where('status', 'active')->latest()->paginate(40);
    }

    /**
     * Render the component view.
     *
     * @return \Illuminate\View\View The view to render.
     */
    public function render()
    {
        return view('blog::post');
    }
}
