<?php

namespace Adfox\ECommerce\Filament\App\Resources\WarehouseResource\RelationManagers;

use Adfox\ECommerce\Models\Product;
use App\Settings\UserSettings;
use Filament\Tables\Actions\Action;
use Filament\Forms;
use Filament\Forms\Components\Actions\Action as ActionsAction;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Tables;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Notifications\Notification;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class InventoryRelationManager extends RelationManager
{
    protected static string $relationship = 'inventories';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('title')
                    ->required()
                    ->maxLength(255),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->defaultSort('created_at', 'desc')
            ->recordTitleAttribute('title')
            ->columns([
                SpatieMediaLibraryImageColumn::make('ad.ads')
                    ->collection('ads')
                    ->conversion('thumb')
                    ->defaultImageUrl(fn($record)=>getAdPlaceholderImage($record->ad?->id))
                    ->label(__('messages.t_ad_images'))
                    ->size(40)
                    ->circular()
                    ->overlap(2)
                    ->stacked()
                    ->limit(3),
                TextColumn::make('ad.title')
                    ->limit(app(UserSettings::class)?->max_character ?? 45)
                    ->tooltip(
                        fn(TextColumn $column): ?string =>
                        strlen($column->getState()) > $column->getCharacterLimit() ? $column->getState() : null
                    )
                    ->label(__('messages.t_ad_title'))
                    ->searchable(),
                TextColumn::make('quantity_in_stock'),
                TextColumn::make('reorder_level'),
            ])
            ->filters([])
            ->headerActions([
                Action::make('Manage Inventory')
                    ->label(__('messages.t_manage_inventory'))
                    ->color('success')
                    ->form([
                        $this->buildProductSelect(),
                        $this->buildQuantityField(),
                        TextInput::make('reorder_level')
                            ->numeric()
                            ->required(),
                    ])
                    ->action(function ($data) {
                        self::processInventoryUpdate($data);
                    }),
            ])
            ->modifyQueryUsing(fn(Builder $query) => $query->latest())
            ->actions([
                Action::make('Manage inventory')
                    ->label(__('messages.t_manage_inventory'))
                    ->color('success')
                    ->form([$this, 'inventoryForm'])
                    ->action(function ($data) {
                        self::processInventoryUpdate($data);
                    }),
            ])
            ->bulkActions([])
            ->emptyStateActions([]);
    }

    protected function buildProductSelect(): Select
    {
        return Select::make('ad_id')
            ->options(Product::where('user_id', auth()->id())->pluck('title', 'id'))
            ->label('Product')
            ->live(onBlur: true)
            ->afterStateUpdated(function (Get $get, Set $set) {
                $inventory = $this->getOwnerRecord()->inventories()
                    ->where('ad_id', $get('ad_id'))
                    ->where('user_id', auth()->id())
                    ->first();

                $set('quantity_in_stock', $inventory->quantity_in_stock ?? 0);
                $set('reorder_level', $inventory->reorder_level ?? null);
            })->required();
    }

    protected function buildQuantityField(): TextInput
    {
        return TextInput::make('quantity_in_stock')
            ->required()
            ->readOnly()
            ->live(onBlur: true)
            ->prefixAction($this->buildRemoveQuantityAction())
            ->suffixAction($this->buildAddQuantityAction());
    }

    protected function buildRemoveQuantityAction(): ActionsAction
    {
        return ActionsAction::make('Remove quantity')
            ->icon('heroicon-m-minus')
            ->form([TextInput::make('remove_quantity')->numeric()->required()])
            ->action(function ($data, Get $get, Set $set) {
                $stock = max(0, $get('quantity_in_stock') - $data['remove_quantity']);
                $set('quantity_in_stock', $stock);
            });
    }

    protected function buildAddQuantityAction(): ActionsAction
    {
        return ActionsAction::make('Add quantity')
            ->icon('heroicon-m-plus')
            ->form([TextInput::make('new_quantity')->numeric()->required()])
            ->action(function ($data, Get $get, Set $set) {
                $currentQuantity = $this->getOwnerRecord()->inventories()->sum('quantity_in_stock');
                $remainingQuantity = $this->getOwnerRecord()->capacity - $currentQuantity;

                if (!isset($data['new_quantity'])) {
                    $data['new_quantity'] = 1;
                }

                if ($data['new_quantity'] > $remainingQuantity) {
                    self::warehouseCapacityNotification($remainingQuantity);

                    return;
                }

                $stock = $get('quantity_in_stock') + $data['new_quantity'];
                $set('quantity_in_stock', $stock);
            });
    }

    public function inventoryForm($record): array
    {
        return [
            TextInput::make('quantity_in_stock')
                ->required()
                ->readOnly()
                ->default($record->quantity_in_stock)
                ->live(onBlur: true)
                ->prefixAction($this->buildRemoveQuantityAction())
                ->suffixAction($this->buildAddQuantityAction()),
            TextInput::make('reorder_level')
                ->default($record->reorder_level)
                ->numeric()
                ->required(),
        ];
    }

    public function processInventoryUpdate($data): void
    {
        $currentQuantity = $this->getOwnerRecord()->inventories()->sum('quantity_in_stock');
        $remainingQuantity = $this->getOwnerRecord()->capacity - $currentQuantity;

        if ($data['quantity_in_stock'] > $remainingQuantity) {
            self::warehouseCapacityNotification($remainingQuantity);

            return;
        }

        $inventory = $this->getOwnerRecord()->inventories()
            ->where('ad_id', $data['ad_id'])
            ->where('user_id', auth()->id())
            ->first();

        $stockTransactions = [
            'ad_id' => $data['ad_id'],
            'user_id' => auth()->id(),
            'quantity' => abs($data['quantity_in_stock'] - ($inventory->quantity_in_stock ?? 0)),
            'transaction_date' => now(),
            'transaction_type' => $inventory ? 'adjustment' : 'purchase',
        ];

        // dd($data);

        if ($inventory) {
            $inventory->update($data);
        } else {
            $data['user_id'] = auth()->id();
            $this->getOwnerRecord()->inventories()->create($data);
        }

        $this->getOwnerRecord()->stock_transactions()->create($stockTransactions);

        Notification::make()
            ->title(__('messages.t_updated_successfully'))
            ->success()
            ->send();
    }

    private function warehouseCapacityNotification($remainingQuantity)
    {
        $capacity = $this->getOwnerRecord()->capacity;
        $remainingQuantity = max(0, $remainingQuantity); // Ensure remainingQuantity is not negative

        Notification::make()
            ->title('Warehouse Capacity Alert')
            ->body("You are nearing your warehouse capacity. Your current capacity is {$capacity} units, with {$remainingQuantity} units of space remaining.")
            ->info()
            ->send();
    }
}
