<?php
namespace Adfox\Subscription\Filament\App\Pages\Promotion;

use App\Models\PlanFeature;
use App\Models\Ad;
use App\Models\AdPromotion;
use App\Models\Promotion;
use Carbon\Carbon;

use Illuminate\Support\Facades\Session;
use Livewire\Attributes\On;
use Livewire\Component;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Illuminate\Support\Facades\Redirect;
use Livewire\Attributes\Url;

class ChoosePromotion extends Page
{
    protected static string $view = 'subscription::promotion.choose-promotion';
    protected static bool $shouldRegisterNavigation = false;

    public $selectedAdSlugs;
    #[Url(as: 'promotion_type', keep: true)]
    public $promotionType;
    #[Url(as: 'id', keep: true)]
    public $id;
    public $promotions = [];
    public $selectedPromotions = [];
    public $availablePlanPromotions = [];
    public $current = 'promotion-details';
    public $steps = [
        'promotion-details',
        'ad.post-ad.payment-ad'
    ];
    public function mount()
    {
        $this->selectedAdSlugs = Session::get('selected-ads');
        if ($this->promotionType == 'apply') {
            $this->loadPlanPromotions();
        }
    }


    public function loadPlanPromotions()
    {
        //Fetch active subscription for the user
        $plans = auth()->user()->getActiveSubscriptions()->pluck('plan_id');
        if ($plans->isEmpty()) {
            return 0;
        }
        //Get promotion id with value
        $promotionValues = PlanFeature::type('promotion')->whereHas('plan', function ($query) use ($plans) {
                $query->whereIn('id', $plans);
            })->selectRaw('sum(value) as ad_count,promotion_id')->groupBy('promotion_id')->pluck('ad_count', 'promotion_id');
            foreach ($promotionValues as $promotionId => $value) {
                $promotion = Promotion::find($promotionId);
                $promotion->total_count = $value;
                if ($this->isEligibleForUpgrade($this->id, $promotionId) && $this->getRemainAdPromotionCount($promotionId)) {
                    $this->availablePlanPromotions[] = $promotion;
                }
            }
    }


    /**
     * Toggle a promotion's selection status.
     *
     * @param int $promotionId The ID of the promotion to toggle.
     */
    public function toggleSelection($promotionId)
    {
        if ($this->promotionType === 'apply') {
            // For 'single' type, only allow one item to be selected at a time.
            $this->selectedPromotions = [$promotionId];
        } else {
            // For other types, allow multiple selections.
            if (in_array($promotionId, $this->selectedPromotions)) {
                // If the item is already selected, deselect it.
                $this->selectedPromotions = array_diff($this->selectedPromotions, [$promotionId]);
            } else {
                // If the item is not selected, add it to the selection.
                $this->selectedPromotions[] = $promotionId;
            }
        }
        $this->dispatch('promotion-selected', $this->selectedPromotions);
    }

    #[On('proceed-payment')]
    public function proceedPayment()
    {

        if (count($this->selectedPromotions)) {
            if ($this->promotionType == 'multiple') {
                $this->current = 'payment-promotion';
            }
        }
    }

    #[On('apply-promotion')]
    public function applyPromotions()
    {
        foreach ($this->selectedPromotions as $key => $promotionId) {
            $promotion = Promotion::find($promotionId);
            //Find the available subscription for the specified promotion
            $subscription = $this->findAvailableSubscription($promotionId);
            $subscriptionId = $subscription?->id ?? null;
            // Check if the ad is eligible for the promotion upgrade and if a subscription ID is available
            if ($this->isEligibleForUpgrade($this->id, $promotionId) && $subscriptionId) {
                //Create AdPromotion with the given details
                $adPromotion = AdPromotion::create([
                    'ad_id' => $this->id,
                    'promotion_id' => $promotionId,
                    'start_date' => Carbon::now(),
                    'end_date' => $subscription?->ends_at,
                    'source' => 'subscription',
                    'subscription_id' => $subscriptionId,
                ]);
                $routeParameters = ['ad_id' => $this->id];
                return Redirect::route('success-upgrade', $routeParameters);

            } else {
                // if promotion is already active return notification
                Notification::make()
                    ->title(trans('messages.t_already_active', ['promotion' => $promotion->name]))
                    ->danger()
                    ->send();
            }
        }
    }

    /**
     * Calculate the remaining ad promotion count for promotion
     *
     * @param [type] $promotionId
     * @return void $adPromotionCount
     */
    public function getRemainAdPromotionCount($promotionId)
    {
        $user = auth()->user();
        $activeSubscriptions = $user->getActiveSubscriptions();
        $adPromotionCount = 0;
        foreach ($activeSubscriptions as $subscription) {
            $remainAdPromotionCount = $subscription->getRemainAdPromotionCount($promotionId);
            $adPromotionCount += $remainAdPromotionCount;
        }
        return $adPromotionCount;
    }

    /**
     * check if ad is eligible for promotion
     *
     * @param [type] $adId
     * @param [type] $promotionId
     * @return boolean
     */
    public function isEligibleForUpgrade($adId, $promotionId)
    {
        $existingPromotion = AdPromotion::where('ad_id', $adId)
            ->where('promotion_id', $promotionId)
            ->latest()
            ->first();
        if (!$existingPromotion || $existingPromotion->end_date->isPast()) {
            return true;
        }
        return false;
    }
    /**
     * Find available subscription for ad promotion based on the ad promotion availability
     *
     * @param [type] $promotionId
     * @return void
     */
    public function findAvailableSubscription($promotionId)
    {
        $user = auth()->user();
        //Fetch active subscription for the user
        $activeSubscriptions = $user->getActiveSubscriptions();
        foreach ($activeSubscriptions as $subscription) {
            $remainAdPromotionCount = $subscription->getRemainAdPromotionCount($promotionId);
            if ($remainAdPromotionCount > 0) {
                return $subscription;
            }
        }
    }

    /**
     * Move to the previous step or redirect to home if at the first step.
     */
    public function redirectBack()
    {
        $currentIndex = array_search($this->current, $this->steps);

        if ($currentIndex === 0) {
            // Redirect to home if at the first step
            return redirect(route('filament.app.pages.my-ads'));
        }

        if ($currentIndex !== false && isset($this->steps[$currentIndex - 1])) {
            $this->current = $this->steps[$currentIndex - 1];
        }
    }

}
