<?php

namespace Adfox\Subscription\Filament\Resources\PlanResource\Pages;

use Adfox\Subscription\Filament\Resources\PlanResource;
use App\Models\Plan;
use App\Settings\StripeSettings;
use Filament\Actions;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\CreateRecord;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Log;
use Stripe\StripeClient;

class CreatePlan extends CreateRecord
{
    use CreateRecord\Concerns\Translatable;

    protected static string $resource = PlanResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\LocaleSwitcher::make(),
            // ...
        ];
    }

    public function getStripePaymentSettingsProperty()
    {
        return app(StripeSettings::class);
    }

    protected function handleRecordCreation(array $data): Model
    {
        $response = $this->createProductAndPrice($data);
        if(isset($data['is_free']) && $data['is_free']){
            $freePlans=Plan::free()->get();
            foreach($freePlans as $freePlan){
                $freePlan->is_free=false;
                $freePlan->save();
            }
        }
        if ($response->isOk()) {

            $productData = $response->getData(true);
            $data = array_merge($data, $productData);

            $plan = static::getModel()::create($data);

            if (isSubscriptionPluginEnabled()) {
                Artisan::call('paypal:initialize-plan');
            }

            return $plan;
        } else {
            $this->halt();
        }
    }

    public function createProductAndPrice($plan)
    {
        try {
            $stripe = $this->initializeStripeClient();
            // Create a new product
            $product = $stripe->products->create([
                'name' => $plan['name'],
                'description' => $plan['description'],
            ]);
            //Create price
            $price = $stripe->prices->create([
                'product' => $product['id'],
                'unit_amount' => (int) $plan['price'] * 100,
                'currency' => $this->stripePaymentSettings?->currency,
                'recurring' => [
                    'interval' => $plan['invoice_interval'],
                    'interval_count' => $plan['invoice_period']
                ],
            ]);

            return response()->json([
                'stripe_product_id' => $product->id,
                'price_id' => $price->id,
            ]);
        } catch (\Stripe\Exception\CardException $e) {
            Notification::make()
                ->title($e->getError()->message)
                ->send();
            return response()->json(['error' => $e->getError()->message], 500);
        } catch (\Exception $e) {
            // Log the unexpected error
            Log::error(`Unexpected error: {$e->getMessage()}`);
            Notification::make()
                ->danger()
                ->body($e->getMessage())
                ->title('Failed to Create Plan')
                ->send();
            // Handle the error gracefully
            return response()->json(['error' => 'An unexpected error occurred. Please try again.'], 500);
        }
    }

    public function initializeStripeClient()
    {
        // Set Stripe API key
        return new StripeClient($this->stripePaymentSettings?->secret_key);
    }
}
