<?php

namespace Adfox\Subscription\Http\Controllers;

use Adfox\Subscription\Notifications\SubscriptionCreatedNotification;
use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\Plan;
use App\Models\Subscription;
use App\Models\User;
use App\Traits\PaypalSubscription;
use Carbon\Carbon;
use Filament\Notifications\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class PaypalPaymentController extends Controller
{
    use PaypalSubscription;

    // Constants for event types
    private const PAYMENT_COMPLETED = 'PAYMENT.SALE.COMPLETED';
    private const SUBSCRIPTION_CREATED = 'BILLING.SUBSCRIPTION.CREATED';
    private const SUBSCRIPTION_PAYMENT_FAILED = 'BILLING.SUBSCRIPTION.PAYMENT.FAILED';
    private const SUBSCRIPTION_ACTIVATED = 'BILLING.SUBSCRIPTION.ACTIVATED';
    private const SUBSCRIPTION_CANCELLED = 'BILLING.SUBSCRIPTION.CANCELLED';

    public function __construct()
    {
        $this->initializeConfig();
    }

    public function webhookHandle(Request $request)
    {
        $eventType = $request->event_type;

        Log::info('PayPal Webhook received', ['event_type' => $eventType]);

        switch ($eventType) {
            case self::SUBSCRIPTION_CREATED:
                $this->handleSubscriptionCreated($request);
                break;
            case self::PAYMENT_COMPLETED:
                $this->handlePaymentSaleCompleted($request);
                break;
            case self::SUBSCRIPTION_ACTIVATED:
                $this->handleSubscriptionActivated($request);
                break;
        }

        return response()->json(['status' => 'success']);
    }

    protected function handleSubscriptionCreated(Request $request)
    {
        $result = collect($request->resource);
        $startDate = $result['start_time'] ? Carbon::parse($result['start_time']) : null;

        $userId = $result['custom_id'];
        $user = User::find($userId);
        $plan = Plan::where('paypal_plan_id', $result['plan_id'])->first();
        $planId = $plan->id;

        if ($user) {
            $subscriptionRecord = $user->subscriptions()->create([
                'plan_id' => $planId,
                'subscription_reference' => $result['id'],
                'payment_method' => 'paypal',
                'starts_at' => $startDate,
                'status' => $result['status'],
                'metadata' => null,
            ]);

            $user->notify(new SubscriptionCreatedNotification($subscriptionRecord));

            return 'Subscription created for user: ' . $user->id;
        }
    }

    protected function handlePaymentSaleCompleted(Request $request)
    {
        $result = collect($request->resource);

        return $this->createPaypalInvoice($result);
    }

    protected function handlePaymentSubscriptionPaymentFailed(Request $request)
    {
        $subscription = $this->findSubscriptionByBillingAgreementId($request->resource['billing_agreement_id']);
        if (!$subscription) {
            Log::info('Subscription not found for billing agreement ID: ' . $request->resource['billing_agreement_id']);
            return;
        }

        $subscription->update(['status' => 'failed']);

        $subscription->histories()->create([
            'transaction_details' => $request->resource,
            'status' => 'failed',
        ]);
    }

    protected function handleSubscriptionActivated(Request $request)
    {
        $result = collect($request->resource);
        $userId = $result['custom_id'];
        $user = User::find($userId);
        $endDate = $result['billing_info']['final_payment_time'] ? Carbon::parse($result['billing_info']['final_payment_time']) : null;

        if ($user) {
            $user->subscriptions()
                ->where('subscription_reference', $result['id'])
                ->first()->update([
                    'ends_at' => $endDate,
                    'status' => $result['status'],
                ]);

            return 'Subscription updated for user: ' . $user->id;
        }
    }

    protected function handleSubscriptionCancelled(Request $request)
    {
        Log::info('Subscription cancelled');
    }

    private function cancelPreviousSubscription(Subscription $currentSubscription)
    {
        $lastSubscription = Subscription::where('status', 'open')->latest()->first();
        if ($lastSubscription?->meta_data) {
            $this->getPayPalClient()->cancelSubscription(
                $lastSubscription->meta_data['id'],
                'Choose another plan'
            );
        }
    }

    private function createPaypalInvoice($result)
    {
        // Step 1: Initialize PayPal provider and get the access token
        $accessToken = $this->provider->getAccessToken();
        // Step 2: Define the invoice data
        $showSubscription = $this->provider->showSubscriptionDetails($result['billing_agreement_id']);
        $plan = $this->provider->showPlanDetails($showSubscription['plan_id']);

        $invoiceData = [
            "merchant_info" => [
                "email" => $showSubscription['subscriber']['email_address'],
                "first_name" => "Merchant",
                "last_name" => "Example",
                "business_name" => env('APP_NAME'),
                "phone" => [
                    "country_code" => "001",
                    "national_number" => "4085551234"
                ],
                "address" => [
                    "line1" => "1234 Main St.",
                    "city" => "San Jose",
                    "state" => "CA",
                    "postal_code" => "95131",
                    "country_code" => "US"
                ]
            ],
            "billing_info" => [[
                "email" => $showSubscription['subscriber']['email_address'],
                "billing_agreement_id" => $showSubscription['id'] // Add billing agreement ID here
            ]],
            "items" => [[
                "name" => $plan['name'],
                "quantity" => 1,
                "unit_price" => [
                    "currency" => $plan['payment_preferences']['setup_fee']['currency_code'],
                    "value" => $plan['payment_preferences']['setup_fee']['value'] // Subscription amount
                ]
            ]],
            "note" => "Invoice for subscription payment.",
            "terms" => "Thank you for subscribing!",
            "shipping_info" => [
                "first_name" => $showSubscription['subscriber']['name']['given_name'],
                "last_name" =>  $showSubscription['subscriber']['name']['surname'],
                "address" => [
                    "line1" => $showSubscription['subscriber']['shipping_address']['address']['address_line_1'],
                    "city" => $showSubscription['subscriber']['shipping_address']['address']['admin_area_2'],
                    "state" => $showSubscription['subscriber']['shipping_address']['address']['admin_area_1'],
                    "postal_code" => $showSubscription['subscriber']['shipping_address']['address']['postal_code'],
                    "country_code" => $showSubscription['subscriber']['shipping_address']['address']['country_code'],
                ]
            ],
        ];

        // Step 3: Create the invoice
        $response = Http::withToken($accessToken['access_token'])
            ->post('https://api.sandbox.paypal.com/v1/invoicing/invoices', $invoiceData);

        if ($response->successful()) {
            $invoice = $response->json();
            $this->provider->registerPaymentInvoice($invoice['id'], Carbon::now()->format('Y-m-d'), 'PAYPAL', "25.00", "Payment received via PayPal");

            $invoice = $this->provider->showInvoiceDetails($invoice['id']);

            $user = User::find($result['custom']);

            $dueDate = $result['create_time'] ? Carbon::parse($result['create_time']) : null;
            $invoiceDate = $result['create_time'] ? Carbon::parse($result['create_time']) : null;
            $subscription = Subscription::where('subscription_reference', $result['billing_agreement_id'])->first();

            if ($user && $subscription) {
                Invoice::create([
                    'subscription_id' => $subscription->id,
                    'invoice_id' => $invoice['id'],
                    'currency' => $invoice['payments']['paid_amount']['currency_code'],
                    'amount_due' => $invoice['payments']['paid_amount']['value'],
                    'amount_paid' => $invoice['payments']['paid_amount']['value'],
                    'status' => $invoice['status'],
                    'amount_remaining' => 0,
                    'invoice_date' => $invoiceDate,
                    'due_date' => $dueDate,
                ]);

                return 'Invoice created for user: ' . $user->id;
            } else {
                return ('User not found' . $user->id);
            }

            return $this->provider->showInvoiceDetails($invoice['id']);
        } else {
            // Handle errors
            return 'Failed to create invoice.';
        }
    }
}
