<?php

namespace Adfox\Subscription\Livewire\Plan;

use App\Models\Plan;
use App\Models\PackageItem;
use App\Settings\PaystackSettings;
use App\Settings\RazorpaySettings;
use Filament\Forms\Components\Radio;
use Livewire\Attributes\Url;
use Livewire\Component;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use App\Settings\StripeSettings;
use App\Settings\PaypalSettings;
use App\Settings\PaymentSettings;
use App\Settings\FlutterwaveSettings;
use App\Settings\OfflinePaymentSettings;
use App\Settings\PaymongoSettings;
use Carbon\Carbon;
use Akaunting\Money\Currency as AkauntingCurrency;
use App\Models\Coupon;
use App\Settings\SubscriptionSettings;
use Filament\Notifications\Notification;

class PlanPayment extends Component implements HasForms
{
    use InteractsWithForms;

    public $currentPayment;
    public $payment_method;
    public $plan;
    public $subtotal = 0;
    public $tax = 0; // Define tax rate if applicable
    public $total = 0;
    public $discountAmount = 0;
    public $isDifferentRate = false;
    public $convertedTotal = 0;
    public $defaultCurrency;
    #[Url]
    public $selectedPlan;
    public $planDetails;
    public $coupons;
    public $couponCode;
    public $appliedCoupon;
    protected $paymentGateways = [
        'stripe' => 'stripe-subscription',
        'paypal' => 'paypal-subscription'
    ];

    /**
     * Initialize the component with the selected plan items and set up payment options.
     *
     * @param array $planDetails Array of selected item IDs.
     */
    public function mount($planDetails)
    {
        $this->planDetails = $planDetails;
        $this->initializePaymentOptions();
        $this->updatePaymentData();
    }

    public function applyCoupon()
    {
        $coupon = Coupon::where('code', $this->couponCode)->first();
        if (!($coupon && $coupon->canBeUsed())) {
            Notification::make()
                ->title(__('messages.t_coupon_invalid'))
                ->danger()
                ->send();
            return;
        }
        $this->appliedCoupon = $coupon;
        $this->updatePaymentData();
    }

    public function removeCoupon($couponId)
    {
        $this->appliedCoupon=null;
    }
    /**
     * Update the payment-related data based on selected plan items.
     */
    protected function updatePaymentData()
    {
        if (isset($this->planDetails['plan_id'])) {
            $this->selectedPlan = $this->planDetails['plan_id'];
        }
        $this->plan = Plan::with('features')->find($this->selectedPlan);
        if (!$this->plan) {
            abort(404);
        }
        $this->subtotal = 0;
        if (isset($this->planDetails['price'])) {
            $this->subtotal = $this->planDetails['price'];
        }

        // Accessing PaymentSettings
        $paymentSettings = app(PaymentSettings::class);

        if ($paymentSettings->enable_tax) {
            if ($paymentSettings->tax_type === 'percentage') {
                // Calculate tax as a percentage of the subtotal
                $this->tax = ($this->subtotal * $paymentSettings->tax_rate) / 100;
            } else if ($paymentSettings->tax_type === 'fixed') {
                // Apply a fixed tax rate
                $this->tax = $paymentSettings->tax_rate;
            }
        } else {
            // No tax applied
            $this->tax = 0;
        }
        $this->tax = 0;
        // Add tax calculation logic here if necessary
        $this->total = $this->subtotal + $this->tax;
        // Apply discounts from multiple coupons
        $this->discountAmount = $this->calculateMultipleDiscounts($this->subtotal);
        // Subtract total discount from total amount
        $this->total = $this->total - $this->discountAmount;

        $this->defaultCurrency = $paymentSettings->currency;
        $paymentGatewayRate = $this->getPaymentGatewayRate();
        $systemExchangeRate = app(PaymentSettings::class)->exchange_rate;

        $this->isDifferentRate = $paymentGatewayRate != 1.0 && $paymentGatewayRate != $systemExchangeRate;
        $this->convertedTotal = $this->total * $paymentGatewayRate / $systemExchangeRate;
    }

    /**
     * Retrieve the current payment gateway exchange rate.
     *
     * @return float The exchange rate of the selected payment gateway.
     */
    private function getPaymentGatewayRate()
    {
        return match ($this->payment_method) {
            'stripe' => app(StripeSettings::class)->exchange_rate,
            'paypal' => app(PaypalSettings::class)->exchange_rate,
            'flutterwave' => app(FlutterwaveSettings::class)->exchange_rate,
            'offline' => app(OfflinePaymentSettings::class)->exchange_rate,
            'paymongo' => app(PaymongoSettings::class)->exchange_rate,
            'razorpay' => app(RazorpaySettings::class)->exchange_rate,
            'paystack' => app(PaystackSettings::class)->exchange_rate,
            default => 1.0
        };
    }

    public function getExchangeCurrencySymbol()
    {
        return match ($this->payment_method) {
            'stripe' => (new AkauntingCurrency(app(StripeSettings::class)->currency))->getSymbol(),
            'paypal' => (new AkauntingCurrency(app(PaypalSettings::class)->currency))->getSymbol(),
            'flutterwave' => (new AkauntingCurrency(app(FlutterwaveSettings::class)->currency))->getSymbol(),
            'offline' => (new AkauntingCurrency(app(OfflinePaymentSettings::class)->currency))->getSymbol(),
            'paymongo' => (new AkauntingCurrency(app(PaymongoSettings::class)->currency))->getSymbol(),
            'razorpay' => (new AkauntingCurrency(app(RazorpaySettings::class)->currency))->getSymbol(),
            'paystack' => (new AkauntingCurrency(app(PaystackSettings::class)->currency))->getSymbol(),
            default => '$'
        };
    }

    /**
     * Initialize available payment options based on system settings.
     *
     * @return array Array of available payment options.
     */
    protected function initializePaymentOptions()
    {
        $paymentOptions = [];

        if (app(StripeSettings::class)->status && app(SubscriptionSettings::class)->status) {
            $paymentOptions['stripe'] = app(StripeSettings::class)->name;
        }

        if (isPaypalPluginEnabled() && isPaypalEnabled() && isSubscriptionPaypalEnabled()) {
            $paymentOptions['paypal'] = app(PaypalSettings::class)->name;
        }

        // Set default payment method if only one option is enabled
        if (count($paymentOptions) === 1) {
            $defaultMethod = array_key_first($paymentOptions);
            $this->payment_method = $defaultMethod;
            $this->currentPayment = $this->paymentGateways[$defaultMethod];
        }

        return $paymentOptions;
    }

    /**
     * Method to get payment data in the specified format.
     *
     * @return array
     */
    public function getPaymentDataProperty()
    {
        $payment = [
            'user_id' => auth()->id(),
            'plan_id' => $this->selectedPlan,
            'price_id' => $this->plan->price_id,
            'subtotal' => $this->subtotal,
            'tax' => $this->tax,
            'total' => $this->total,
            'ad_count' => 0,
            'featured_ad_count' => 0,
            'urgent_ad_count' => 0,
            'spotlight_ad_count' => 0,
            'website_url_count' => 0
        ];
        if (isset($this->planDetails['ad_count'])) {
            $payment['ad_count'] = $this->planDetails['ad_count'];
        }
        if (isset($this->planDetails['featured_ad_count'])) {
            $payment['featured_ad_count'] = $this->planDetails['featured_ad_count'];
        }
        if (isset($this->planDetails['urgent_ad_count'])) {
            $payment['urgent_ad_count'] = $this->planDetails['urgent_ad_count'];
        }
        if (isset($this->planDetails['spotlight_ad_count'])) {
            $payment['spotlight_ad_count'] = $this->planDetails['spotlight_ad_count'];
        }
        if (isset($this->planDetails['website_url_count'])) {
            $payment['website_url_count'] = $this->planDetails['website_url_count'];
        }
        if( $this->appliedCoupon?->stripe_coupon_id){
            $payment['coupon_id'] = $this->appliedCoupon->stripe_coupon_id;
        }
        return $payment;
    }

    /**
     * Define the form schema for payment method selection.
     *
     * @param Form $form The filament form instance.
     * @return Form The configured form schema.
     */
    public function form(Form $form): Form
    {
        $paymentOptions = $this->initializePaymentOptions();

        return $form
            ->schema([
                Radio::make('payment_method')
                    ->hiddenLabel()
                    ->live()
                    ->options($paymentOptions)
                    ->afterStateUpdated(function ($state) {
                        $this->currentPayment = $this->paymentGateways[$state] ?? null;
                        $this->updatePaymentData();
                    }),
            ]);
    }

    public function getCoupons()
    {
        $this->coupons = Coupon::where('is_active', true)->all();
    }

    /**
     * Calculate total discount from multiple coupons on a given subtotal.
     *
     * @param float $subtotal
     * @return float
     */
    protected function calculateMultipleDiscounts(float $subtotal): float
    {
        $totalDiscount = 0;

        if($this->appliedCoupon){
        $totalDiscount+= $this->calculateDiscount($this->appliedCoupon,$subtotal);
}
        return $totalDiscount;
    }

    public function calculateDiscount($coupon, $subtotal)
    {
        $totalDiscount = 0;
        // Calculate the discount for the coupon based on its type
        if ($coupon->type === 'fixed') {
            $totalDiscount += min($coupon->discount_value, $subtotal - $totalDiscount); // Prevent over-discounting
        } elseif ($coupon->type === 'percentage') {
            $totalDiscount += ($coupon->discount_value / 100) * ($subtotal - $totalDiscount); // Apply percentage on remaining subtotal
        }
        return $totalDiscount;
    }

    // public function getPriceId(){
    //     //Create price
    //     $price = $stripe->prices->create([
    //         'product' => $product['id'],
    //         'unit_amount' => (int) $plan['price'] * 100,
    //         'currency' => $this->stripePaymentSettings?->currency,
    //         'recurring' => [
    //             'interval' => $plan['invoice_interval'],
    //             'interval_count' => $plan['invoice_period']
    //         ],
    //     ]);
    // }
    /**
     * Render the component view.
     *
     * @return \Illuminate\View\View The view to render.
     */
    public function render()
    {
        return view('subscription::plan.plan-payment');
    }
}
