<?php
namespace Adfox\Subscription\Services;

use App\Http\Controllers\Logger;
use Adfox\Subscription\Notifications\SubscriptionCreatedNotification;
use Adfox\Subscription\Notifications\SubscriptionDeletedNotification;
use Adfox\Subscription\Notifications\SubscriptionUpdatedNotification;
use App\Models\Subscription;
use App\Models\User;
use Carbon\Carbon;

class SubscriptionService
{
    /**
     * Logs a message to a custom logger.
     *
     * @param string $message The message to log.
     * @param array  $context Additional log context.
     */
    private function logMessage($type, $text = '', $timestamp = true)
    {
        $logger = new Logger(storage_path() . '/logs/subscription.log');
        $logger->log($type, $text, $timestamp);
    }

    /**
     * HANDLE CUSTOMER.SUBSCRIPTION.CREATED EVENT
     *
     *   create subscription record for that subscriber(user)
     *   send subscription created email and notification to the user
     *
     * @param [type] $subscription
     * @return void
     */
    public function handleSubscriptionCreated($subscription)
    {
        $userId = $subscription->metadata->user_id;
        $planId = $subscription->metadata->plan_id;

        //Ad Count
        $ad_count = $subscription->metadata->ad_count ?? 0;
        $feature_ad_count = $subscription->metadata->feature_ad_count ?? 0;
        $urgent_ad_count = $subscription->metadata->urgent_ad_count ?? 0;
        $spotlight_ad_count = $subscription->metadata->spotlight_ad_count ?? 0;
        $website_url_count = $subscription->metadata->website_url_count ?? 0;

        $startDate = $subscription->start_date ? Carbon::createFromTimestamp($subscription->start_date) : null;
        $endDate = $subscription->current_period_end ? Carbon::createFromTimestamp($subscription->current_period_end) : null;
        $cancelDate = $subscription->cancel_at ? Carbon::createFromTimestamp($subscription->cancel_at) : null;
        // Access the first item in the subscription's items array
        $subscriptionItem = $subscription->items->data[0];

        // Get the price details
        $price = $subscriptionItem->price;

        // Access price details
        $unitAmount = $price->unit_amount;
        $formattedPrice = $unitAmount / 100;
        $user = User::find($userId);

        if ($user) {
            $existingSubscription=$user->getActiveSubscriptions()->first();
            if($existingSubscription){
                $existingSubscription->status='paused';
                $existingSubscription->save();
            }
            $subscriptionRecord = $user->subscriptions()->create([
                'plan_id' => $planId,
                'subscription_reference' => $subscription->id,
                'payment_method' => 'stripe',
                'starts_at' => $startDate,
                'ends_at' => $endDate,
                'cancels_at' => $cancelDate,
                'status' => $subscription->status,
                'metadata' => $subscription->metadata,
                'price' => $formattedPrice,
                'ad_count' => $ad_count,
                'feature_ad_count' => $feature_ad_count,
                'urgent_ad_count' => $urgent_ad_count,
                'spotlight_ad_count' => $spotlight_ad_count,
                'website_url_count' => $website_url_count,
            ]);

            $user->notify(new SubscriptionCreatedNotification($subscriptionRecord));

            $this->logMessage('Subscription created for user: ' . $user->id);
        }
    }

    /**
     * HANDLE CUSTOMER.SUBSCRIPTION.UPDATED EVENT
     *
     *   update status,cancels_at subscription record with that latest information
     *   send subscription updated email and notification to the user
     *
     * @param [type] $subscription
     * @return void
     */
    public function handleSubscriptionUpdated($subscription)
    {
        $userId = $subscription->metadata->user_id;
        $user = User::find($userId);
        $cancelDate = $subscription->cancel_at ? Carbon::createFromTimestamp($subscription->cancel_at) : null;
        $subscriptionRecord = Subscription::where('subscription_reference', $subscription->id)->first();

        $subscriptionRecord->update([
            'status' => $subscription->status,
            'cancels_at' => $cancelDate,
        ]);

        if ($user && $subscriptionRecord)
            $user->notify(new SubscriptionUpdatedNotification($subscriptionRecord));

        $this->logMessage('Subscription updated for user: ' . $userId);
    }

    /**
     * HANDLE CUSTOMER.SUBSCRIPTION.DELETED EVENT
     *
     *   update ends_at,status,cancels_at subscription record with that latest information
     *   send subscription cancel or deleted email and notification to the user
     *
     * @param [type] $subscription
     * @return void
     */
    public function handleSubscriptionDeleted($subscription)
    {
        $userId = $subscription->metadata->user_id;
        $user = User::find($userId);
        $endDate = $subscription->current_period_end ? Carbon::createFromTimestamp($subscription->current_period_end) : null;
        $cancelDate = $subscription->canceled_date ? Carbon::createFromTimestamp($subscription->canceled_date) : null;

        $subscription = Subscription::where('subscription_reference', $subscription->id)->update([
            'ends_at' => $endDate,
            'status' => $subscription->status,
            'cancels_at' => $cancelDate,
        ]);

        if ($user)
            $user->notify(new SubscriptionDeletedNotification($subscription));

        if ($subscription)
            $this->logMessage('Subscription deleted for user: ' . $userId);
    }
}
