<?php

namespace Adfox\WalletSystem\Filament\App\Resources;

use Adfox\WalletSystem\Filament\App\Resources\BankAccountResource\Pages;
use App\Models\Wallets\BankAccount;
use App\Models\Wallets\Wallet;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Forms\Set;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ToggleColumn;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Str;

class BankAccountResource extends Resource
{
    protected static ?string $model = BankAccount::class;
    protected static ?int $navigationSort = 2;

    public static function canViewAny(): bool
    {
        $totalWallet = Wallet::where('user_id', auth()->id())->first();

        return max($totalWallet?->balance, 0) > 1 && is_ecommerce_active();
    }

    public static function getNavigationGroup(): string
    {
        return __('messages.t_wallet_system_navigation');
    }

    public static function getNavigationLabel(): string
    {
        return __('messages.t_bank_account_label');
    }

    public static function getModelLabel(): string
    {
        return __('messages.t_bank_account_label');
    }

    //TODO: Enable when wallet system is implemented
    // public static function isDiscovered(): bool
    // {
    //     return !is_vehicle_rental_active();
    // }
    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Select::make('bank_id')
                    ->relationship('bank', 'bank_name')
                    ->label(__('messages.t_bank_account_table_columns.bank_name'))
                    ->required(),

                TextInput::make('account_number')
                    ->numeric()
                    ->label(__('messages.t_bank_account_table_columns.account_number'))
                    ->minLength(10)
                    ->required(),

                TextInput::make('ifsc_code')
                    ->rules(['regex:/^[a-zA-Z0-9]+$/'])
                    ->live(onBlur: true)
                    ->afterStateUpdated(fn(Set $set, ?string $state) => $set('ifsc_code', Str::upper($state)))
                    ->maxLength(11)
                    ->label(__('messages.t_bank_account_table_columns.ifsc_code'))
                    ->required(),

                TextInput::make('account_holder_name')
                    ->label(__('messages.t_bank_account_table_columns.account_holder_name'))
                    ->required(),

                Select::make('status')
                    ->label(__('messages.t_bank_account_table_columns.status'))
                    ->options([
                        'active' => __('messages.t_bank_account_status.active'),
                        'inactive' => __('messages.t_bank_account_status.inactive'),
                    ])
                    ->required(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultSort('created_at', 'desc')
            ->emptyState(view('tables.empty-state', ['message' => __('messages.t_no_bank_account_details')]))
            ->modifyQueryUsing(fn(Builder $query) => $query->where('user_id', auth()->id()))
            ->columns([
                TextColumn::make('bank.bank_name')
                    ->label(__('messages.t_bank_account_table_columns.bank_name'))
                    ->sortable()
                    ->searchable(),

                TextColumn::make('account_number')
                    ->label(__('messages.t_bank_account_table_columns.account_number'))
                    ->sortable()
                    ->searchable(),

                TextColumn::make('ifsc_code')
                    ->label(__('messages.t_bank_account_table_columns.ifsc_code'))
                    ->sortable()
                    ->searchable(),

                TextColumn::make('account_holder_name')
                    ->label(__('messages.t_bank_account_table_columns.account_holder_name'))
                    ->sortable()
                    ->searchable(),

                TextColumn::make('status')
                    ->label(__('messages.t_bank_account_table_columns.status'))
                    ->badge()
                    ->colors([
                        'success' => 'active',
                        'danger'  => 'inactive',
                    ])
                    ->formatStateUsing(fn($state) => __('messages.t_bank_account_status.' . $state))
                    ->sortable()
                    ->searchable(),

                ToggleColumn::make('is_default')
                    ->label(__('messages.t_bank_account_table_columns.is_default'))
                    ->sortable()
                    ->beforeStateUpdated(function ($record, $state) {
                        BankAccount::where('user_id', auth()->id())->each(function ($item) {
                            $item->update(['is_default' => false]);
                        });
                    })
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListBankAccounts::route('/'),
            'create' => Pages\CreateBankAccount::route('/create'),
            'edit' => Pages\EditBankAccount::route('/{record}/edit'),
        ];
    }
}
