<?php

namespace Adfox\Blog\Filament\Resources;

use Adfox\Blog\Filament\Resources\PostCategoryResource\Pages;
use Adfox\Blog\Filament\Resources\PostCategoryResource\RelationManagers;
use App\Filament\Clusters\Blog;
use App\Models\PostCategory;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Forms\Set;
use Filament\Resources\Concerns\Translatable;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Str;
use BezhanSalleh\FilamentShield\Contracts\HasShieldPermissions;


class PostCategoryResource extends Resource implements HasShieldPermissions
{
    use Translatable;

    protected static ?string $model = PostCategory::class;

    protected static ?string $slug = 'blog/categories';

    protected static ?string $cluster = Blog::class;

    protected static ?string $navigationIcon = 'heroicon-o-folder';

    protected static ?string $recordTitleAttribute = 'name';

    public static function getPermissionPrefixes(): array
    {
        return [
            'create',
            'update',
            'view_any',
            'delete_any',
        ];
    }


    public static function canViewAny(): bool
    {
        return userHasPermission('view_any_post::category');
    }

    public static function canCreate(): bool
    {
        return userHasPermission('create_post::category');
    }

    public static function canEdit($record): bool
    {
        return userHasPermission('update_post::category');
    }

    public static function canDeleteAny(): bool
    {
        return userHasPermission('delete_post::category');
    }

    public static function getModelLabel(): string
    {
        return __('messages.t_ap_post_category');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('name')
                ->label(__('messages.t_ap_name'))
                ->unique(ignoreRecord: true)
                ->live(onBlur:true)
                ->afterStateUpdated(fn (Set $set, ?string $state) => $set('slug', Str::slug($state)))
                ->required()
                ->maxLength(255),

            Forms\Components\Textarea::make('description')
                ->label(__('messages.t_ap_description'))
                ->required(),

            Forms\Components\TextInput::make('order')
                ->label(__('messages.t_ap_order'))
                ->required()
                ->maxLength(255),

            Forms\Components\TextInput::make('slug')
                ->label(__('messages.t_ap_category_url'))
                ->unique(ignoreRecord: true),

            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultSort('created_at', 'desc')
            ->columns([
                TextColumn::make('name')
                ->label(__('messages.t_ap_name'))

            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPostCategories::route('/'),
            'create' => Pages\CreatePostCategory::route('/create'),
            'edit' => Pages\EditPostCategory::route('/{record}/edit'),
        ];
    }
}
