<?php

namespace Adfox\Blog\Filament\Resources;

use Adfox\Blog\Filament\Resources\PostResource\Pages;
use App\Filament\Clusters\Blog;
use App\Models\BlogPost;
use App\Models\PostCategory;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Infolists\Components;
use Filament\Infolists\Infolist;
use Filament\Notifications\Notification;
use Filament\Pages\SubNavigationPosition;
use Filament\Resources\Pages\Page;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;
use BezhanSalleh\FilamentShield\Contracts\HasShieldPermissions;


class PostResource extends Resource implements HasShieldPermissions
{
    protected static ?string $model = BlogPost::class;
    protected static ?string $slug = 'blog/posts';
    protected static ?string $recordTitleAttribute = 'title';
    protected static ?int $navigationSort = 0;

    protected static ?string $cluster = Blog::class;

    protected static ?string $navigationIcon = 'heroicon-o-pencil-square';

    protected static SubNavigationPosition $subNavigationPosition = SubNavigationPosition::Top;

    public static function getPermissionPrefixes(): array
    {
        return [
            'create',
            'update',
            'view_any',
            'delete_any',
            'view',
            'delete'
        ];
    }

    public static function canViewAny(): bool
    {
        return userHasPermission('view_any_post');
    }

    public static function canView($record): bool
    {
        return userHasPermission('view_post');
    }

    public static function canCreate(): bool
    {
        return userHasPermission('create_post');
    }

    public static function canEdit($record): bool
    {
        return userHasPermission('update_post');
    }

    public static function canDelete($record): bool
    {
        return userHasPermission('delete_post');
    }

    public static function canDeleteAny(): bool
    {
        return userHasPermission('delete_any_post');
    }

    public static function getModelLabel(): string
    {
        return __('messages.t_ap_post');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make()
                    ->schema([
                        Forms\Components\TextInput::make('title')
                            ->label(__('messages.t_ap_title'))
                            ->required()
                            ->live(onBlur: true)
                            ->maxLength(100)
                            ->afterStateUpdated(fn(string $operation, $state, Forms\Set $set) => $operation === 'create' ? $set('slug', Str::slug($state)) : null),

                        Forms\Components\TextInput::make('slug')
                            ->label(__('messages.t_ap_slug'))
                            ->disabled()
                            ->dehydrated()
                            ->required()
                            ->maxLength(255)
                            ->unique(BlogPost::class, 'slug', ignoreRecord: true),

                        Forms\Components\MarkdownEditor::make('content')
                            ->label(__('messages.t_ap_content'))
                            ->required()
                            ->columnSpan('full'),

                        Forms\Components\DatePicker::make('published_at')
                            ->label(__('messages.t_ap_published_date')),

                        Forms\Components\Select::make('category_id')
                            ->label(__('messages.t_ap_category'))
                            ->options(PostCategory::all()->pluck('name', 'id')),
                    ])
                    ->columns(2),

                Forms\Components\Section::make(__('messages.t_ap_seo_details'))
                    ->relationship('seo')
                    ->schema([
                        Forms\Components\TextInput::make('title')
                            ->label(__('messages.t_ap_seo_title'))
                            ->required(),
                        Forms\Components\Textarea::make('description')
                            ->label(__('messages.t_ap_seo_description'))
                            ->required(),
                    ]),

                Forms\Components\Section::make(__('messages.t_ap_image'))
                    ->schema([
                        Forms\Components\SpatieMediaLibraryFileUpload::make('image')
                            ->label(__('messages.t_ap_image_upload'))
                            ->maxSize(maxUploadFileSize())
                            ->collection('blogs')
                            ->visibility('public')
                            ->image()
                            ->required()
                            ->imageEditor(),
                    ])
                    ->collapsible(),

            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultSort('created_at', 'desc')
            ->emptyState(view('tables.empty-state', ['message' => __('messages.t_no_post')]))
            ->columns([
                Tables\Columns\SpatieMediaLibraryImageColumn::make('ads')
                    ->collection('blogs')
                    ->defaultImageUrl(asset('images/placeholder.jpg'))
                    ->label(__('messages.t_ap_blog_image'))
                    ->size(40),

                Tables\Columns\TextColumn::make('title')
                    ->label(__('messages.t_ap_title'))
                    ->searchable()
                    ->sortable(),

                Tables\Columns\TextColumn::make('slug')
                    ->label(__('messages.t_ap_slug'))
                    ->searchable()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                Tables\Columns\TextColumn::make('status')
                    ->label(__('messages.t_ap_status'))
                    ->badge()
                    ->getStateUsing(fn(BlogPost $record): string => $record->published_at?->isPast() ? __('messages.t_ap_published') : __('messages.t_ap_draft'))
                    ->colors([
                        'success' => __('messages.t_ap_published'),
                    ]),

                Tables\Columns\TextColumn::make('published_at')
                    ->label(__('messages.t_ap_published_date'))
                    ->date(),

            ])
            ->filters([
                Tables\Filters\Filter::make('published_at')
                    ->label(__('messages.t_ap_published_date_filter'))
                    ->form([
                        Forms\Components\DatePicker::make('published_from')
                            ->label(__('messages.t_ap_published_from'))
                            ->placeholder(fn($state): string => __('messages.t_ap_placeholder_published_from', ['year' => now()->subYear()->format('Y')])),
                        Forms\Components\DatePicker::make('published_until')
                            ->placeholder(fn($state): string => now()->format('M d, Y')),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['published_from'] ?? null,
                                fn(Builder $query, $date): Builder => $query->whereDate('published_at', '>=', $date),
                            )
                            ->when(
                                $data['published_until'] ?? null,
                                fn(Builder $query, $date): Builder => $query->whereDate('published_at', '<=', $date),
                            );
                    })
                    ->indicateUsing(function (array $data): array {
                        $indicators = [];
                        if ($data['published_from'] ?? null) {
                            $indicators['published_from'] = __('messages.t_ap_indicator_published_from', [
                                'date' => Carbon::parse($data['published_from'])->toFormattedDateString(),
                            ]);
                        }
                        if ($data['published_until'] ?? null) {
                            $indicators['published_until'] = __('messages.t_ap_indicator_published_until', [
                                'date' => Carbon::parse($data['published_until'])->toFormattedDateString(),
                            ]);
                        }

                        return $indicators;
                    }),

            ])
            ->actions([
                Tables\Actions\ViewAction::make(),

                Tables\Actions\EditAction::make(),

                Tables\Actions\DeleteAction::make(),
            ])
            ->groupedBulkActions([
                Tables\Actions\DeleteBulkAction::make()
            ]);
    }

    public static function infolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->schema([
                Components\Section::make()
                    ->schema([
                        Components\Split::make([
                            Components\Grid::make(2)
                                ->schema([
                                    Components\Group::make([
                                        Components\TextEntry::make('title')
                                        ->label(__('messages.t_ap_title')),
                                        Components\TextEntry::make('slug')
                                        ->label(__('messages.t_ap_slug')),
                                        Components\TextEntry::make('published_at')
                                        ->label(__('messages.t_ap_published_date'))
                                            ->badge()
                                            ->date()
                                            ->color('success'),
                                    ])
                                ]),
                            Components\SpatieMediaLibraryImageEntry::make('image')
                                ->collection('blogs')
                                ->hiddenLabel()
                                ->grow(false),
                        ])->from('lg'),
                    ]),
                Components\Section::make(__('messages.t_ap_section_content'))
                    ->schema([
                        Components\TextEntry::make('content')
                            ->prose()
                            ->markdown()
                            ->hiddenLabel(),
                    ])
                    ->collapsible(),
            ]);
    }

    public static function getRecordSubNavigation(Page $page): array
    {
        return $page->generateNavigationItems([
            Pages\ViewPost::class,
            Pages\EditPost::class,
            Pages\ManagePostComments::class,
        ]);
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPosts::route('/'),
            'create' => Pages\CreatePost::route('/create'),
            'comments' => Pages\ManagePostComments::route('/{record}/comments'),
            'edit' => Pages\EditPost::route('/{record}/edit'),
            'view' => Pages\ViewPost::route('/{record}'),
        ];
    }
}
