<?php

namespace Adfox\ECommerce\Livewire\Product\Post;

use Adfox\ECommerce\Models\ReturnPolicy;
use App\Foundation\AdBase\BaseAdDetail;
use App\Models\Ad;
use App\Settings\OfflinePaymentSettings;
use Closure;
use Filament\Forms\Components\MarkdownEditor;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Forms\Get;

class ProductFieldDetail extends BaseAdDetail
{
    public $sku;
    public $return_policy_id;
    public $enable_cash_on_delivery;

    protected function fillAdDetails()
    {
        $this->title = $this->ad->title;
        $this->description = $this->ad->description;
        $this->price = $this->ad->price;
        $this->type = $this->ad->type;
        $this->for_sale_by = $this->ad->for_sale_by;
        $this->tags = $this->ad->tags ? $this->ad->tags : [];
        $this->price_type_id = $this->ad->price_type_id;
        $this->display_phone = $this->ad->display_phone;
        $this->phone_number = $this->ad->phone_number;
        $this->condition_id = $this->ad->condition_id;
        $this->price_suffix = $this->ad->price_suffix;
        $this->offer_price = $this->ad->offer_price;
        $this->sku = $this->ad->sku;
        $this->return_policy_id = $this->ad->return_policy_id;
        $this->enable_cash_on_delivery = $this->ad->enable_cash_on_delivery;
        
        if ($this->ad->category_id) {
            $this->category_id = $this->ad->category_id;
            if ($this->ad->category) {
                $this->parent_category = $this->ad->category->parent_id;
                $this->showMainCategories = false;
                $this->disable_condition = $this->ad->adType?->disable_condition;
                $this->disable_price_type = $this->ad->adType?->disable_price_type;
            }
        }
    }

    public function getRequiredFieldsProperty()
    {
        $requiredFields = [];
        $rules = $this->getRules();

        foreach ($rules as $field => $rule) {
            if (is_array($rule) && in_array('required', $rule)) {
                $requiredFields[] = $field;
            } elseif ($rule == 'required') {
                $requiredFields[] = $field;
            }
        }
        return $requiredFields;
    }

    public function detailForm(): Form
    {
        return $this->makeForm()
            ->schema([
                TextInput::make('price')
                    ->label(__('messages.t_price_your_ad'))
                    ->helperText(__('messages.t_set_fair_price'))
                    ->required()
                    ->validationAttribute(__('messages.t_set_fair_price'))
                    ->numeric()
                    ->minValue(1)
                    ->placeholder(__('messages.t_price_your_ad'))
                    ->prefix(config('app.currency_symbol'))
                    ->live(onBlur: true),
                TextInput::make('offer_price')
                    ->label(__('messages.t_offer_price'))
                    ->helperText(__('messages.t_set_offer_price'))
                    ->validationAttribute(__('messages.t_offer_price'))
                    ->numeric()
                    ->minValue(1)
                    ->lt('price')
                    ->placeholder(__('messages.t_enter_offer_price'))
                    ->prefix(config('app.currency_symbol'))
                    ->live(onBlur: true),
                ...self::getEcommerceDetailForm(),
                TagsInput::make('tags')
                    ->label(__('messages.t_tags'))
                    ->helperText(__('messages.t_set_tags'))
                    ->live(onBlur: true),
            ]);
    }

    public function checkRequiredFieldsFilled()
    {
        $isFilled = false;
        foreach ($this->requiredFields as $field) {
            if (trim($this->$field) !== '' && (!empty(trim($this->$field))) && (!is_null(trim($this->$field)))) {
                $isFilled = true;
            } else {
                $isFilled = false;
                break;
            }
        }

        if (isset($this->display_phone) && $this->display_phone && (!empty($this->phone_number))) {
            $isFilled = $isFilled ? true : false;
        }
        $this->dispatch('required-fields-filled', isFilled: $isFilled);
    }

    public function getEcommerceDetailForm()
    {
        return [
            MarkdownEditor::make('description')
                ->disableToolbarButtons([
                    'attachFiles'
                ])
                ->label(__('messages.t_description'))
                ->live(onBlur: true)
                ->minLength(20)
                ->maxLength(300)
                ->required(),
            TextInput::make('sku')
                ->rules([
                    fn(Get $get): Closure => function (string $attribute, $value, Closure $fail) use ($get) {
                        $ad = Ad::where('sku', $value)->exists();

                        if ($ad && $this->ad->sku != $value) {
                            $fail("The {$attribute} is already exists.");
                        }
                    },
                ])
                ->maxLength(20)
                ->label(__('messages.t_sku'))
                ->helperText(__('messages.t_sku_helper_text'))
                ->required()
                ->validationAttribute(__('messages.t_sku'))
                ->minLength(5)
                ->placeholder(__('messages.t_sku'))
                ->live(onBlur: true),
            Select::make('return_policy_id')
                ->label(__('messages.t_return_policy'))
                ->helperText(__('messages.t_sku_helper_text'))
                ->required()
                ->live(onBlur: true)
                ->options(ReturnPolicy::where('user_id', auth()->id())->pluck('policy_name', 'id')),
            Toggle::make('enable_cash_on_delivery')
                ->label(__('messages.t_enable_cash_on_delivery'))
                ->helperText(__('messages.t_enable_cash_on_delivery_helper_text'))
                ->required()
                ->visible(app('filament')->hasPlugin('offline-payment') && app(OfflinePaymentSettings::class)->status)
                ->validationAttribute(__('messages.t_enable_cash_on_delivery'))
        ];
    }

    public function businessSpecificForm(): Form
    {
        // Regular ads don't have business-specific fields
        return $this->makeForm()->schema([
            TextInput::make('price')
                ->label(__('messages.t_price_your_ad'))
                ->helperText(__('messages.t_set_fair_price'))
                ->required()
                ->validationAttribute(__('messages.t_set_fair_price'))
                ->numeric()
                ->minValue(1)
                ->placeholder(__('messages.t_price_your_ad'))
                ->prefix(config('app.currency_symbol'))
                ->live(onBlur: true),
            TextInput::make('offer_price')
                ->label(__('messages.t_offer_price'))
                ->helperText(__('messages.t_set_offer_price'))
                ->validationAttribute(__('messages.t_offer_price'))
                ->numeric()
                ->minValue(1)
                ->lt('price')
                ->placeholder(__('messages.t_enter_offer_price'))
                ->prefix(config('app.currency_symbol'))
                ->live(onBlur: true),
            ...self::getEcommerceDetailForm(),
            TagsInput::make('tags')
                ->label(__('messages.t_tags'))
                ->helperText(__('messages.t_set_tags'))
                ->live(onBlur: true),
        ]);
    }

    public function updated($name, $value)
    {
        $this->checkRequiredFieldsFilled();
        $userId = auth()->id();
        $this->validateOnly($name);
        if (!$this->id) {
            if (!$userId) {
                abort(403, 'Unauthorized action.');
            }
            $ad = Ad::create([$name => $value, 'user_id' => $userId]);
            $this->id = $ad->id;
            $this->updateAdSlug($ad, $value);
            $this->loadAdDetails($this->id);
        } else {
            $ad = Ad::find($this->id);

            if (!$ad || $ad->user_id != $userId) {
                abort(403, 'Unauthorized action.');
            }
            // Update the "tags" property if it's an array element update
            if (str_starts_with($name, 'tags.')) {
                $index = explode('.', $name)[1];
                $tags = $this->tags;
                $tags[$index] = $value;
                $this->tags = $tags;
                $ad->update(['tags' => json_encode($tags)]);
            } else {
                // Update other fields
                $adData = [$name => $value];
                $ad->update($adData);
                if ($name == 'title') {
                    // If the title is updated, update the slug too
                    $this->updateAdSlug($ad->fresh(), $value);
                }
                if ($name == 'category_id') {
                    $this->disable_condition = $ad->adType?->disable_condition;
                    $this->disable_price_type = $ad->adType?->disable_price_type;

                    // If the sub category is updated, update the location if it has default location
                    $this->updateLocation($ad, $value);
                }
            }
        }
        $this->dispatch('ad-created', id: $ad->id);
    }

    public function save()
    {
        $this->validate();

        $adData = $this->only([
            'title',
            'category_id',
            'description',
            'price',
            'price_type_id',
            'condition_id',
            'display_phone',
            'phone_number',
            'for_sale_by',
            'tags',
            'price_suffix',
            'offer_price'
        ]);

        if ($this->id) {
            $ad = Ad::find($this->id);
            $ad->update($adData);
        } else {
            $ad = Ad::create(array_merge($adData, ['user_id' => auth()->id()]));
        }

        $this->updateAdSlug($ad, $this->title);

        $this->dispatch('ad-saved', id: $ad->id);
        return $this->redirect('/ads');
    }
}
