<?php

namespace Adfox\Paymongo\Livewire;

use App\Models\WebhookPackage;
use Livewire\Component;
use App\Models\WebhookUpgrade;
use App\Settings\PaymentSettings;
use App\Settings\PaymongoSettings;
use App\Settings\GeneralSettings;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Session;

class PaymongoPayment extends Component
{

    public $id;

    public $type;

    public $total;

    public $data;

    public $tax;

    public $subtotal;

    public $successUrl;

    public $payment_gateway_params = [];

    /**
     * Mount the component and set the properties if an ad ID is provided.
     */
    public function mount($id)
    {
        $this->id = $id;
        $this->processPayment();
        $this->successUrl=url('callback/paymongo');
    }

    public function getPaymentSettingsProperty()
    {
        return app(PaymentSettings::class);
    }

    public function getGeneralSettingsProperty()
    {
        return app(GeneralSettings::class);
    }

    public function getPaymongoSettingsProperty()
    {
        return app(PaymongoSettings::class);
    }

    public function processPayment()
    {

        // Generate order id for PayPal
        $order_id = $this->type . '-' . uid(18);

        // Set PayPal payment parameters
        $this->payment_gateway_params['order_id'] = $order_id;


        $this->handleWebhookUpgrade($order_id);
    }

    protected function handleWebhookUpgrade($order_id)
    {
        try {
            if ($this->type == 'PKG') {
                WebhookPackage::create([
                    'data' => json_encode($this->data),
                    'payment_id' => $order_id,
                    'payment_method' => 'paymongo',
                    'status' => 'pending'
                ]);
            } else {
                WebhookUpgrade::create([
                    'data' => json_encode($this->data),
                    'payment_id' => $order_id,
                    'payment_method' => 'paymongo',
                    'status' => 'pending'
                ]);
            }
        } catch (\Throwable $th) {
            // Handle any exceptions
        }
    }


    public function makePaymongoPayment()
    {
        // Assuming $this->total is in pesos, convert to centavos by multiplying by 100
        $amountInCentavos = $this->total * 100;

        // Check if the amount is below the minimum required by PayMongo
        if ($amountInCentavos < 2000) {
            Notification::make()
                ->title('The total amount cannot be less than PHP 20.00.')
                ->danger() // This might need to be changed to ->error() or similar, depending on the context and available methods
                ->send();
            return; // Make sure to exit the function if the condition is met
        }

        $client = new \GuzzleHttp\Client();

        try {
            $response = $client->request('POST', 'https://api.paymongo.com/v1/checkout_sessions', [
                'body' => json_encode([
                    'data' => [
                        'attributes' => [
                            'send_email_receipt' => false,
                            'show_description' => true,
                            'show_line_items' => false,
                            'line_items' => [
                                [
                                    'currency' => 'PHP',
                                    'amount' => $amountInCentavos,
                                    'description' => $this->payment_gateway_params['order_id'],
                                    'quantity' => 1,
                                    'name' => 'Item Name',
                                ],
                            ],
                            'description' => $this->type == 'UG' ? 'Ad Upgrades' : 'Package Purchase',
                            'payment_method_types' => ['card', 'dob', 'dob_ubp','gcash', 'grab_pay','paymaya'],
                            'reference_number' => $this->payment_gateway_params['order_id'],
                            'success_url' => $this->successUrl,
                        ],
                    ],
                ]),
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                    'Authorization' => 'Basic ' . $this->paymongoSettings?->authorization_token,
                ],
            ]);

            $data = json_decode($response->getBody(), true);

            if(is_array($data)){
                Session::put('paymongo_checkout_id',$data['data']['id']);
                return redirect()->to($data['data']['attributes']['checkout_url']);
            }

        } catch (\GuzzleHttp\Exception\ClientException $e) {
            // Here, you can catch and handle the exception
            $response = $e->getResponse();
            $responseBodyAsString = $response->getBody()->getContents();

            // Consider using Notification to show error message
            Notification::make()
                ->title('Error initiating payment: ' . $responseBodyAsString)
                ->danger()
                ->send();
        }
    }

    /**
     * Render the component view.
     *
     * @return \Illuminate\View\View The view to render.
     */
    public function render()
    {
        return view('paymongo::paymongo-payment');
    }
}
