<?php

namespace Adfox\Paystack\Livewire;

use App\Models\WebhookPackage;
use App\Traits\HandlesPackageApplication;
use App\Traits\HandlesPromotionApplication;
use Filament\Notifications\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Livewire\Attributes\On;
use Livewire\Component;
use App\Models\WebhookUpgrade;
use App\Settings\PaymentSettings;
use App\Settings\PaystackSettings;
use App\Settings\GeneralSettings;

class PaystackPayment extends Component
{
    use HandlesPackageApplication;
    use HandlesPromotionApplication;
    public $id;

    public $type;

    public $total;

    public $data;

    public $tax;

    public $subtotal;

    public $successUrl;

    public $payment_gateway_params = [];

    /**
     * Mount the component and set the properties if an ad ID is provided.
     */
    public function mount($id)
    {
        $this->id = $id;
        $this->successUrl = url('callback/paystack');
        $this->processPayment();
    }

    public function getPaymentSettingsProperty()
    {
        return app(PaymentSettings::class);
    }

    public function getGeneralSettingsProperty()
    {
        return app(GeneralSettings::class);
    }

    public function getPaystackSettingsProperty()
    {
        return app(PaystackSettings::class);
    }

    public function processPayment()
    {
        $orderId = $this->type . '-' . uid(18);
        $url = "https://api.paystack.co/transaction/initialize";

        //Payment information
        $fields = [
            'email' => auth()->user()->email,
            'amount' => $this->total*100,
            'callback_url' => $this->successUrl,
            'metadata' => ["order_id" => $orderId],
            'currency' => $this->paystackSettings->currency
        ];

        $headers = [
            "Cache-Control: no-cache",
        ];

        //Make http request to initialize the payment
        $response = Http::withToken($this->paystackSettings->secret_key)->asForm()
            ->withHeaders($headers)
            ->post($url, $fields);

        //Handle successful response
        if ($response->successful()) {
            //Get the response data object
            $data = $response->object();
            //handle the webhook upgrade
            $this->handleWebhookUpgrade($orderId);
            //redirect to the payment page
            return redirect()->to($data->data->authorization_url);
        } else {
            // Consider using Notification to show error message
            Notification::make()
                ->title('Error initiating payment: ' . $response->object()?->message)
                ->danger()
                ->send();
        }
    }


    protected function handleWebhookUpgrade($order_id)
    {
        try {
            if ($this->type == 'PKG') {
                WebhookPackage::create([
                    'data' => json_encode($this->data),
                    'payment_id' => $order_id,
                    'payment_method' => 'paystack',
                    'status' => 'pending'
                ]);
            } else {
                WebhookUpgrade::create([
                    'data' => json_encode($this->data),
                    'payment_id' => $order_id,
                    'payment_method' => 'paystack',
                    'status' => 'pending'
                ]);
            }
        } catch (\Throwable $th) {
            // Handle any exceptions
        }
    }

    /**
     * Render the component view.
     *
     * @return \Illuminate\View\View The view to render.
     */
    public function render()
    {
        return view('paystack::paystack-payment');
    }
}
