<?php

namespace Adfox\Phonepe\Http\Controllers\Callback;

use App\Http\Controllers\Controller;
use App\Models\WebhookPackage;
use App\Models\WebhookUpgrade;
use App\Settings\PhonePeSettings;
use App\Traits\HandlesPromotionApplication;
use App\Traits\HandlesPackageApplication;
use Illuminate\Support\Str;
use PhonePe\PhonePe;

class PhonePeController extends Controller
{
    use HandlesPackageApplication;
    use HandlesPromotionApplication;

    private PhonePeSettings $phonePeSettings;
    private PhonePe $phonePe;

    private const PAYMENT_SUCCESS = 'PAYMENT_SUCCESS';
    private const STATUS_PENDING = 'pending';

    /**
     * Handle the callback from the PhonePe payment gateway.
     *
     * @param string $transactionId
     * @return \Illuminate\Http\RedirectResponse
     */
    public function callback(string $transactionId)
    {
        $this->initializePhonePeSettings();

        $this->phonePe = PhonePe::init(
            $this->phonePeSettings->merchantId,
            $this->getMerchantUserId(),
            $this->phonePeSettings->saltKey,
            $this->phonePeSettings->saltIndex,
            route('phone-pe.redirect', $transactionId),
            route('phone-pe.callback', $transactionId),
            $this->phonePeSettings->env_mode
        );

        $response = $this->verifyPayment($transactionId);

        if ($this->isPaymentSuccessful($response)) {
            return $this->handleWebhookEvent($transactionId);
        }

        return redirect('/');
    }

    /**
     * Initialize PhonePe settings.
     */
    private function initializePhonePeSettings(): void
    {
        $this->phonePeSettings = app(PhonePeSettings::class);
    }

    /**
     * Get merchant user ID.
     *
     * @return string
     */
    private function getMerchantUserId(): string
    {
        return auth()->check() ? (string)auth()->id() : 'MUID' . rand(100000, 999999);
    }

    /**
     * Verify if payment succeeded.
     *
     * @param string $transactionId
     * @return array
     */
    private function verifyPayment(string $transactionId): array
    {
        try {
            $checkout = $this->phonePe->standardCheckout();
            $transactionStatus = $checkout->getTransactionStatusByTransactionId($transactionId);

            if ($transactionStatus === self::PAYMENT_SUCCESS) {
                return [
                    'success'  => true,
                    'response' => $checkout->getTransactionResponseByTransactionId($transactionId)
                ];
            }

            return ['success' => false, 'message' => __('messages.t_error_payment_failed')];
        } catch (\Throwable $th) {
            return ['success' => false, 'message' => __('messages.t_toast_something_went_wrong')];
        }
    }

    /**
     * Check if payment is successful.
     *
     * @param array $response
     * @return bool
     */
    private function isPaymentSuccessful(array $response): bool
    {
        return is_array($response) && $response['success'] === true;
    }

    /**
     * Handle webhook event based on transaction type.
     *
     * @param string $transactionId
     * @return \Illuminate\Http\RedirectResponse
     */
    private function handleWebhookEvent(string $transactionId)
    {
        if (Str::startsWith($transactionId, 'PKG-')) {
            return $this->handlePackageCallback($transactionId);
        }

        return $this->handleUpgradeCallback($transactionId);
    }

    /**
     * Handle package management callback.
     *
     * @param string $transactionId
     * @return \Illuminate\Http\RedirectResponse
     */
    private function handlePackageCallback(string $transactionId)
    {
        $orderData = WebhookPackage::where('payment_id', $transactionId)
            ->where('payment_method', 'phonepe')
            ->where('status', self::STATUS_PENDING)
            ->firstOrFail();

        $routeParameters = $this->applyPackages($orderData);

        return redirect()->route('package-success', $routeParameters);
    }

    /**
     * Handle promotion upgrade callback.
     *
     * @param string $transactionId
     * @return \Illuminate\Http\RedirectResponse
     */
    private function handleUpgradeCallback(string $transactionId)
    {
        $orderData = WebhookUpgrade::where('payment_id', $transactionId)
            ->where('payment_method', 'phonepe')
            ->where('status', self::STATUS_PENDING)
            ->firstOrFail();

        $routeParameters = $this->applyPromotionsWithRedirect($orderData);

        return redirect()->route('success-upgrade', $routeParameters);
    }
}
