<?php

namespace Adfox\Subscription\Filament\Resources\PlanResource\Pages;

use Adfox\Subscription\Filament\Resources\PlanResource;
use App\Models\Plan;
use App\Models\Subscription as ModelsSubscription;
use Filament\Actions;
use Filament\Resources\Pages\EditRecord;
use App\Settings\StripeSettings;
use Filament\Notifications\Notification;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;
use Stripe\StripeClient;
use Stripe\Stripe;
use Stripe\Subscription;

class EditPlan extends EditRecord
{
    use EditRecord\Concerns\Translatable;
    protected static string $resource = PlanResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\DeleteAction::make(),
            Actions\LocaleSwitcher::make(),
        ];
    }

    protected function handleRecordUpdate(Model $record, array $data): Model
    {
        if(isset($data['is_free']) && $data['is_free']){
            $freePlans=Plan::free()->get();
            foreach($freePlans as $freePlan){
                $freePlan->is_free=false;
                $freePlan->save();
            }
        }
        if ($data['price'] != $record->price) {
            $response = $this->updateProductAndPrice($data, $record);
            if ($response->isOk()) {
                $productData = $response->getData(true);
                $data = array_merge($data, $productData);
                $this->archiveOldPrice($record);
                $record->update($data);
            } else {
                $this->halt();
            }
        }else{
            $record->update($data);
        }
        return $record;
    }
    public function updateProductAndPrice($data, $record)
    {
        try {
            $stripe = $this->initializeStripeClient();
            //update the product details
            if (isset($record->stripe_product_id) && $record->stripe_product_id) {
                $product = $stripe->products->update($record->stripe_product_id, [
                    'name' => $data['name'],
                    'description' => $data['description'],
                ]);
            } else {
                //create product
                $product = $stripe->products->create([
                    'name' => $data['name'],
                    'description' => $data['description'],
                ]);
            }
            //create new price
            if ($product) {
                $price = $stripe->prices->create([
                    'product' => $product['id'],
                    'unit_amount' => (int) $data['price'] * 100,
                    'currency' => $this->stripePaymentSettings?->currency,
                    'recurring' => [
                        'interval' => $data['invoice_interval'],
                        'interval_count' => $data['invoice_period']
                    ],
                ]);
                $planId = $record->id;
                $newPriceId = $price->id;
                $this->updateSubscription($planId, $newPriceId);
            }

            return response()->json([
                'stripe_product_id' => $product->id,
                'price_id' => $price->id,
            ]);
        } catch (\Stripe\Exception\CardException $e) {
            Notification::make()
                ->title($e->getError()->message)
                ->send();
            return response()->json(['error' => $e->getError()->message], 500);

        } catch (\Exception $e) {
            // Log the unexpected error
            Log::error(`Unexpected error: {$e->getMessage()}`);
            Notification::make()
                ->danger()
                ->body($e->getMessage())
                ->title('Failed to Update Plan')
                ->send();
            // Handle the error gracefully
            return response()->json(['error' => 'An unexpected error occurred. Please try again.'], 500);
        }

    }
    public function updateSubscription($planId, $newPriceId)
    {
        $subscriptions = ModelsSubscription::where('plan_id', $planId)->pluck('subscription_reference');
        foreach ($subscriptions as $subscriptionId) {
            $response = $this->updateSubscriptionPrice($subscriptionId, $newPriceId);
        }

    }
    public function updateSubscriptionPrice($subscriptionId, $newPriceId)
    {
        // Set your Stripe secret key
        Stripe::setApiKey($this->stripePaymentSettings?->secret_key);

        try {
            // Retrieve the subscription
            $subscription = Subscription::retrieve($subscriptionId);

            // Update the subscription to use the new price
            $updatedSubscription = Subscription::update($subscriptionId, [
                'items' => [
                    [
                        'id' => $subscription->items->data[0]->id,
                        'price' => $newPriceId, // New Price ID
                    ],
                ],
                'proration_behavior' => 'none', // Prorate the difference (optional)
            ]);
            return response()->json(['success' => 'Subscription updated successfully.', 'subscription' => $updatedSubscription], 200);
        } catch (\Exception $e) {
            // Handle the exception
            return response()->json(['error' => $e->getMessage()], 400);
        }

    }
    public function getStripePaymentSettingsProperty()
    {
        return app(StripeSettings::class);
    }
    public function initializeStripeClient()
    {
        return new StripeClient($this->stripePaymentSettings?->secret_key);
    }

    public function archiveOldPrice($record)
    {
        try{
        $stripe = $this->initializeStripeClient();
        // disable old price id
        if (isset($record->price_id) && $record->price_id) {
            $archivedPrice = $stripe->prices->update($record->price_id, [
                'active' => false,
            ]);
        }
    } catch (\Exception $e) {
        // Notification::make()
        //     ->title($e->getError()->message)
        //     ->send();
        Log::error(`Unexpected error: {$e->getMessage()}`);
        return response()->json(['error' => $e->getMessage()], 500);

    }
    }
}
